<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:receipt="http://www.e-dec.ch/xml/schema/edecReceiptResponse/v2">

    <!--document language-->
    <xsl:variable name="documentLanguage" select="//receipt:documentInformation/receipt:documentLanguage"/>
    <xsl:variable name="documentTitle" select="//receipt:documentInformation/receipt:documentTitle"/>
    <xsl:variable name="documentType" select="//receipt:documentInformation/receipt:documentType"/>

    <!--Main Templates-->
    <xsl:output method="html" encoding="UTF-8"/>

    <xsl:template match="/">
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:taxationDecisionExport"/>
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:taxationDecisionVAT"/>
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:taxationDecisionCustomsDuties"/>
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:refundVAT"/>
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:refundCustomsDuties"/>
        <xsl:apply-templates select="//receipt:receiptRequestResponse/receipt:receiptList"/>
    </xsl:template>

    <!-- taxationDecision (HTML Version for the 3 PDF Documents from receiptResponse)-->
    <xsl:template match="receipt:receiptRequestResponse/receipt:taxationDecisionExport" name="taxationDecisionExport">
        <xsl:call-template name="mainTemplate"/>
    </xsl:template>
    <xsl:template match="receipt:receiptRequestResponse/receipt:taxationDecisionVAT" name="taxationDecisionVAT">
        <xsl:call-template name="mainTemplate"/>
    </xsl:template>
    <xsl:template match="receipt:receiptRequestResponse/receipt:taxationDecisionCustomsDuties" name="taxationDecisionCustomsDuties">
        <xsl:call-template name="mainTemplate"/>
    </xsl:template>
    <xsl:template match="receipt:receiptRequestResponse/receipt:refundVAT" name="refundVAT">
        <xsl:call-template name="mainTemplate"/>
    </xsl:template>
    <xsl:template match="receipt:receiptRequestResponse/receipt:refundCustomsDuties" name="refundCustomsDuties">
        <xsl:call-template name="mainTemplate"/>
    </xsl:template>

    <!-- Main Layout for all Documents  -->
    <xsl:template name="mainTemplate">
        <xsl:variable name="declarationType" select="receipt:declarationType"/>
        <xsl:variable name="correctionCode" select="receipt:correctionCode"/>
        <xsl:variable name="acceptanceDate" select="receipt:acceptanceDate"/>
        <xsl:variable name="acceptanceTime" select="receipt:acceptanceTime"/>
        <xsl:variable name="documentDate" select="receipt:documentInformation/receipt:documentDate"/>
        <xsl:variable name="documentTime" select="receipt:documentInformation/receipt:documentTime"/>

        <html content="text/html; charset=UTF-8">
            <!-- HEAD -->
            <head>
                <xsl:call-template name="header_settings"/>
            </head>

            <!-- BODY -->
            <body>
                <!-- header images-->
                <table width="787" border="0">
                    <tr>
                        <td width="100%" scope="row">
                            <xsl:call-template name="header"/>
                        </td>
                    </tr>
                </table>

                <!-- barcode-->
                <table width="787" border="0">
                    <tr>
                        <td width="550" scope="row">
                        </td>
                        <td width="223" rowspan="4">
                            <xsl:call-template name="barcode"/>
                        </td>
                    </tr>

                    <!-- document title-->
                    <tr>
                        <td scope="row">
                            <div align="left">
                                <span class="title">
                                    <xsl:value-of select="$documentTitle"/>
                                </span>
                            </div>
                        </td>
                    </tr>

                    <!-- correction Code-->
                    <xsl:if test="$correctionCode = 0">
                        <tr>
                            <td scope="row">
                                <div align="left">
                                    <span class="subtitle">
                                        <xsl:call-template name="Translate">
                                            <xsl:with-param name="labelToTranslate" select="'annullation'"/>
                                        </xsl:call-template>
                                    </span>
                                </div>
                            </td>
                        </tr>
                    </xsl:if>

                    <!-- datum -->
                    <tr>
                        <td scope="row">
                            <div align="left">
                                <table width="100%" class="content" border="0">
                                    <!-- optional on refund -->
                                    <xsl:if test="$acceptanceDate">
                                        <tr>
                                            <td scope="row" class="label" width="30%">
                                                <xsl:call-template name="Translate">
                                                    <xsl:with-param name="labelToTranslate" select="'annahmedatum'"/>
                                                </xsl:call-template>
                                            </td>
                                            <td>
                                                <xsl:value-of select="$acceptanceDate"/>
                                                <xsl:text>, </xsl:text>
                                                <xsl:value-of select="$acceptanceTime"/>
                                            </td>
                                        </tr>
                                    </xsl:if>
                                    <tr>
                                        <td scope="row" class="label">
                                            <xsl:call-template name="Translate">
                                                <xsl:with-param name="labelToTranslate" select="'ausstellungsdatum'"/>
                                            </xsl:call-template>
                                        </td>
                                        <td>
                                            <xsl:value-of select="$documentDate"/>
                                            <xsl:text>, </xsl:text>
                                            <xsl:value-of select="$documentTime"/>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </td>
                    </tr>

                    <!-- customsOffice-->
                    <tr>
                        <td scope="row">
                            <div align="left">
                                <xsl:call-template name="customsOffice"/>
                            </div>
                        </td>
                    </tr>
                </table>

                <!-- TaxationDecision VAT and CustomsDuties-->
                <xsl:if test="($documentType = 'taxationDecisionVAT')">
                    <xsl:call-template name="taxationDecisionTemplate"/>
                </xsl:if>
                <xsl:if test="($documentType = 'taxationDecisionCustomsDuties')">
                    <xsl:call-template name="taxationDecisionTemplate"/>
                </xsl:if>
                <xsl:if test="($documentType = 'taxationDecisionExport')">
                    <xsl:call-template name="taxationDecisionTemplate"/>
                </xsl:if>

                <!-- refund VAT and CustomsDuties-->
                <xsl:if test="($documentType = 'refundVAT')">
                    <xsl:call-template name="refundTemplate"/>
                </xsl:if>
                <xsl:if test="($documentType = 'refundCustomsDuties')">
                    <xsl:call-template name="refundTemplate"/>
                </xsl:if>

                <!-- legalAdvisory -->
                <table width="787" border="0">
                    <xsl:for-each select="receipt:legalAdvisory">
                        <xsl:call-template name="legalAdvisory"/>
                    </xsl:for-each>
                </table>

                <!-- End of BODY -->
            </body>
            <!-- End of HTML -->
        </html>
    </xsl:template>
    <!-- End of Main Template -->


    <!-- Barcode -->
    <xsl:template name="barcode">
        <xsl:variable name="documentNumber" select="receipt:documentInformation/receipt:documentNumber"/>
        <xsl:variable name="documentVersion" select="receipt:documentInformation/receipt:documentVersion"/>
        <xsl:variable name="documentID">
            <xsl:value-of select="$documentNumber"/>.<xsl:value-of select="$documentVersion"/>
        </xsl:variable>

        <table border="0" align="right" class="content" width="304">
            <tr>
                <td align="right">
                    <xsl:element name="img">
                        <xsl:attribute name="height">50</xsl:attribute>
                        <xsl:attribute name="src">barcode/<xsl:value-of select="$documentID"/>
                        </xsl:attribute>
                    </xsl:element>
                </td>
            </tr>
            <tr id="docNumber">
                <td align="right">
                    <xsl:value-of select="$documentID"/>
                </td>
            </tr>
        </table>
    </xsl:template>

    <!-- Table for Adresses and Goods Items-->
    <xsl:template name="taxationDecisionTemplate">

        <p>
            <xsl:text/>
        </p>
        <table width="787" border="0">
            <tr>
                <td width="513" scope="row" align="left">
                    <!-- versender -->
                    <xsl:if test="receipt:consignor">
                        <xsl:call-template name="consignor"/>
                        <br/>
                    </xsl:if>
                    <xsl:if test="receipt:importer">
                        <xsl:call-template name="importer"/>
                        <br/>
                    </xsl:if>
                    <!-- empf&#228;nger -->
                    <xsl:if test="receipt:consignee">
                        <xsl:call-template name="consignee"/>
                        <br/>
                    </xsl:if>
                    <xsl:if test="receipt:declarant">
                        <xsl:call-template name="spediteur"/>
                        <br/>
                    </xsl:if>
                    <!-- transporteur -->
                    <xsl:if test="receipt:carrier">
                        <xsl:call-template name="carrier"/>
                    </xsl:if>
                </td>
                <td valign="top">
                    <xsl:call-template name="headerInformationRight"/>
                </td>
            </tr>
        </table>

        <!-- Goods Item -->
        <table width="787" border="0">
            <p>
		        <xsl:text>
		        </xsl:text>
            </p>
            <xsl:for-each select="receipt:goodsItem">
                <xsl:call-template name="goodsItem"/>
            </xsl:for-each>
            <br/>
            <br/>
            <br/>
        </table>
    </xsl:template>

    <!-- Table for Adresses and valuationSummary-->
    <xsl:template name="refundTemplate">
        <xsl:variable name="accountHolder" select="receipt:accountHolder"/>
        <!-- konto mwst -->
        <xsl:variable name="VATAccount" select="receipt:valuationSummary/receipt:VAT/receipt:VATAccount"/>
        <xsl:variable name="accountTypeVAT" select="$VATAccount/receipt:accountType"/>
        <xsl:variable name="accountNumberVAT" select="$VATAccount/receipt:accountNumber"/>
        <xsl:variable name="accountNameVAT" select="$VATAccount/receipt:accountName"/>
        <!-- konto zoll -->
        <xsl:variable name="customsAccount" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount"/>
        <xsl:variable name="accountTypeCustoms" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountType"/>
        <xsl:variable name="accountNumberCustoms"
                      select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountNumber"/>
        <xsl:variable name="accountNameCustoms" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountName"/>
        <xsl:variable name="VATNumber" select="receipt:valuationSummary/receipt:VAT/receipt:VATNumber"/>
        <xsl:variable name="traderReference" select="receipt:traderReference"/>
        <xsl:variable name="correctionReason" select="receipt:correctionReason"/>
        <xsl:variable name="customsReference" select="receipt:customsReference"/>
        <xsl:variable name="bordereauNumber" select="receipt:bordereauNumber"/>
        <xsl:variable name="personInCharge" select="receipt:personInCharge"/>

        <p>
            <xsl:text></xsl:text>
        </p>
        <table width="787" border="0">
            <tr>
                <td width="513" scope="row" align="left" class="content">
                    <!-- kunde (opt)-->
                    <xsl:if test="$accountHolder">
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'kunde'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:call-template name="accountHolder"/>
                    </xsl:if>
                    <!-- VATAccount -->
                    <xsl:if test="$VATAccount">
                        <br></br>
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'konto_mwst'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$accountNumberVAT"/>
                        <xsl:text>  </xsl:text>
                        <xsl:value-of select="$accountNameVAT"/>
                        <br></br>
                    </xsl:if>
                    <!-- customsAccount -->
                    <xsl:if test="$customsAccount">
                        <br></br>
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'konto_zoll'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$accountNumberCustoms"/>
                        <xsl:text>  </xsl:text>
                        <xsl:value-of select="$accountNameCustoms"/>
                        <br></br>
                    </xsl:if>
                    <!-- MWST-Nr = VATNumber (mand) -->
                    <xsl:if test="$VATNumber">
                        <br></br>
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'mwst_nr'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$VATNumber"/>
                        <br></br>
                    </xsl:if>
                </td>

                <td valign="top">
                    <!-- sachbearbeiter (mand) -->
                    <table width="100%" class="content" border="0" cellspacing="10">
                        <tr>
                            <td class="label">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'sachbearbeiter'"/>
                                </xsl:call-template>
                            </td>
                            <td>
                                <xsl:value-of select="$personInCharge"/>
                            </td>
                        </tr>

                        <!-- borderau nummer (opt)-->
                        <xsl:if test="$bordereauNumber and string-length($bordereauNumber)>0">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'bordereauNumber'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$bordereauNumber"/>
                                </td>
                            </tr>
                        </xsl:if>

                        <!-- Referenz Zoll (opt)-->
                        <xsl:if test="$customsReference and string-length($customsReference)>0">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'referenz_zoll'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$customsReference"/>
                                </td>
                            </tr>
                        </xsl:if>

                        <!-- Berichtigungsgrund (opt)-->
                        <xsl:if test="$correctionReason and string-length($correctionReason)>0">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'berichtigungsgrund'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$correctionReason"/>
                                </td>
                            </tr>
                        </xsl:if>

                        <!-- Dossiernummer Spediteur (opt)-->
                        <xsl:if test="$traderReference and string-length($traderReference)>0">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'dossiernummer_spediteur'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$traderReference"/>
                                </td>
                            </tr>
                        </xsl:if>

                    </table>
                </td>
            </tr>
        </table>

        <table width="787" border="0" class="table_1">
            <!-- valuationSummary refundVAT -->
            <xsl:if test="($documentType = 'refundVAT')">
                <xsl:call-template name="valuationSummaryRefundVAT"/>
            </xsl:if>
            <!-- valuationSummary refundCustomsDuties -->
            <xsl:if test="($documentType = 'refundCustomsDuties')">
                <xsl:call-template name="valuationSummaryCustomDuties"/>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="valuationSummaryCustomDuties">
        <xsl:variable name="detailAmount" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:detailAmount"/>
        <xsl:variable name="totalAmount" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:totalAmount"/>

        <tr>
            <td class="label" align="center">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'einnahmeart'"/>
                </xsl:call-template>
            </td>
            <td class="label" align="center">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'betrag'"/>
                </xsl:call-template>
            </td>
        </tr>
        <xsl:for-each select="$detailAmount">
            <!-- Stammdaten AdditionalTaxes -->
            <xsl:variable name="type" select="receipt:type"/>
            <xsl:variable name="additionalTaxesTypeText">
                <xsl:choose>
                    <xsl:when test="$type = '150'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'andereGebuehren'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '151'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'edelmetallkontrol'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '280'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'monopolgebuehren'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '290'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'grenztieraerztliche_untersuchung'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '291'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'kanzleigebuehr'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '292'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'fauna-gebuehr'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '430'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'provuebersiedlung'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '450'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tabaksteuer'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '460'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'banderolensteuer'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '465'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'SOTA-gebuehren'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '470'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tabakpraeventionsfonds'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '480'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'biersteuer'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '600'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteue_benzin'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '601'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_diesel'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '602'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_petrol'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '603'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_anderen_treibstoffen'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '605'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'minoestflugbenzin'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '606'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'minoestflugpetrol'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '620'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_benzin'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '621'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_diesel'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '622'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_petrol'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '623'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_anderen_treibstoffen'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '625'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'minoestzusflugbenz'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '626'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'minoestzusflugpetrol'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '640'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_mineraloelprodukten'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '660'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'automobilsteuer'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '700'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_VOC'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '710'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'lenkungsabgaben_heizoel'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '720'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_autobenzin'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '730'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_dieseloel'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '740'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_heizoel'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '741'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_erdgas'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '742'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_mineraloelen'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '743'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_kohle'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '790'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'zollanteil_pflanzenschutzabgabe'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '791'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'pflanzschutzgeb'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '970'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'vorgezogene_entsorgungsgebuehr'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '110'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'zollabgaben'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$type = '995'">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'zollbeguenstkontgeb'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="$type"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>

            <tr>
                <td class="td_content" align="left">
                    <xsl:value-of select="$additionalTaxesTypeText"/>
                </td>
                <td class="td_content" align="right">
                    <xsl:value-of select="receipt:amount"/>
                </td>
            </tr>
        </xsl:for-each>
        <tr>
            <td class="td_content" align="left">
                <b>
                    <xsl:if test="($documentType = 'refundCustomsDuties')">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'nettorueckverguetung'"/>
                        </xsl:call-template>
                    </xsl:if>
                    <xsl:if test="($documentType = 'taxationDecisionCustomsDuties')">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'gesamtbetrag_ohne'"/>
                        </xsl:call-template>
                    </xsl:if>
                </b>
            </td>
            <td class="td_content" align="right">
                <b>
                    <xsl:value-of select="$totalAmount"/>
                </b>
            </td>
        </tr>
    </xsl:template>

    <xsl:template name="valuationSummaryRefundVAT">
        <xsl:variable name="totalAmount" select="receipt:valuationSummary/receipt:VAT/receipt:totalAmount"/>
        <tr>
            <td class="td_label" align="center">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'einnahmeart'"/>
                </xsl:call-template>
            </td>
            <td class="td_label" align="center">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'betrag'"/>
                </xsl:call-template>
            </td>
        </tr>
        <tr>
            <td align="left" class="td_label">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'mwst_rueck'"/>
                </xsl:call-template>
            </td>
            <td align="right" class="td_content">
                <xsl:value-of select="$totalAmount"/>
            </td>
        </tr>
        <tr>
            <td align="left" class="td_label">
                <xsl:call-template name="Translate">
                    <xsl:with-param name="labelToTranslate" select="'nettorueckverguetung'"/>
                </xsl:call-template>
            </td>
            <td align="right" class="td_content">
                <b>
                    <xsl:value-of select="$totalAmount"/>
                </b>
            </td>
        </tr>
    </xsl:template>

    <xsl:template name="header_settings">
        <title>
            <xsl:value-of select="$documentTitle"/>
        </title>
        <style type="text/css">
            <xsl:text>
            .title {font-family:Arial, Helvetica, sans-serif; font-size: 20px; font-style:bold}
            .subtitle {font-family:Arial, Helvetica, sans-serif; font-size: 38px; font-style:bold; color:red}
            .label {font-family:Arial, Helvetica, sans-serif;font-size: 12px; font-style:italic}
            .content {font-family:Arial, Helvetica, sans-serif;font-size: 12px}
            .barcodeTop {font-family:Arial, Helvetica, sans-serif;font-size: 14px}
            .barcodeCenter {font-family:Arial, Helvetica, sans-serif;font-size: 40px}
            .barcodeBottom {font-family:Arial, Helvetica, sans-serif;font-size: 24px}
            .header {font-family:Arial, Helvetica, sans-serif;font-size: 10px}
            .table_0 {border-width: 0px; border-spacing:0px; border-style:solid; border-collapse: collapse;}
            .table_1 {border-style: solid; border-width: 1px 1px 1px 1px; border-spacing: 0; border-collapse: collapse; page-break-inside: avoid}
            .td_label{border-style: solid; margin: 0; padding: 1px;  border-width: 1px 1px 1px 1px;font-family:Arial, Helvetica, sans-serif;font-size: 12px; font-style:italic}
            .td_content{border-style: solid; margin: 0; padding: 1px;  border-width: 1px 1px 1px 1px;font-family:Arial, Helvetica, sans-serif;font-size: 12px}
      	   </xsl:text>
        </style>
    </xsl:template>

    <!-- Child Templates -->
    <xsl:template name="header">
        <table width="100%" border="0">
            <tr>
                <td width="70%" scope="row">
                    <div align="left">
                        <img src="logo_switzerland.gif"/>
                    </div>

                </td>
                <td width="30%" valign="top" class="header">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'eidgenoessisches_finanzdepartement_efd'"/>
                    </xsl:call-template>
                    <br/>
                    <strong>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'eidgenoessische_zollverwaltung_ezv'"/>
                        </xsl:call-template>
                    </strong>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:text> </xsl:text>
                </td>
                <td>
                    <xsl:text> </xsl:text>
                </td>
                <td>
                    <xsl:text> </xsl:text>
                </td>
                <td>
                    <xsl:text> </xsl:text>
                </td>
                <td>
                    <xsl:text> </xsl:text>
                </td>
            </tr>
        </table>
    </xsl:template>

    <xsl:template name="customsOffice">
        <xsl:variable name="customsOfficeNumber" select="receipt:customsOffice/receipt:customsOfficeNumber"/>
        <xsl:variable name="name" select="receipt:customsOffice/receipt:name"/>
        <xsl:variable name="street" select="receipt:customsOffice/receipt:street"/>
        <xsl:variable name="country" select="receipt:customsOffice/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:customsOffice/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:customsOffice/receipt:city"/>
        <xsl:variable name="addressSupplement1" select="receipt:customsOffice/receipt:addressSupplement1"/>
        <xsl:variable name="addressSupplement2" select="receipt:customsOffice/receipt:addressSupplement2"/>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <span class="label">
                        <xsl:value-of select="$customsOfficeNumber"/>:
                    </span>
                    <br></br>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:value-of select="$street"/>
                </td>
            </tr>
            <xsl:if test="$addressSupplement1 and string-length($addressSupplement1)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement1"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$addressSupplement2 and string-length($addressSupplement2)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement2"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:if test="$country and string-length($country)>0">
                        <xsl:value-of select="$country"/>
                        <xsl:text> </xsl:text>
                    </xsl:if>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>
        </table>
    </xsl:template>

    <!-- Kunde -->
    <xsl:template name="accountHolder">
        <xsl:variable name="country" select="receipt:accountHolder/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:accountHolder/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:accountHolder/receipt:city"/>
        <xsl:variable name="accountHolder" select="receipt:accountHolder"/>

        <table width="100%" class="content" border="0">
            <xsl:if test="$accountHolder and string-length($accountHolder)>0">
                <tr>
                    <td>
                        <xsl:for-each select="//receipt:accountHolder/receipt:addressLine">
                            <xsl:value-of select="."/>
                            <br></br>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:value-of select="$country"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>
        </table>
    </xsl:template>

    <xsl:template name="consignor">
        <xsl:variable name="name" select="receipt:consignor/receipt:name"/>
        <xsl:variable name="street" select="receipt:consignor/receipt:street"/>
        <xsl:variable name="country" select="receipt:consignor/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:consignor/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:consignor/receipt:city"/>
        <xsl:variable name="addressSupplement1" select="receipt:consignor/receipt:addressSupplement1"/>
        <xsl:variable name="addressSupplement2" select="receipt:consignor/receipt:addressSupplement2"/>
        <xsl:variable name="traderIdentificationNumber" select="receipt:consignor/receipt:traderIdentificationNumber"/>
        <xsl:variable name="consignorReference" select="receipt:consignor/receipt:consignorReference"/>
        <span class="label">
            <xsl:call-template name="Translate">
                <xsl:with-param name="labelToTranslate" select="'versender'"/>
            </xsl:call-template>
        </span>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:value-of select="$street"/>
                </td>
            </tr>
            <xsl:if test="$addressSupplement1 and string-length($addressSupplement1)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement1"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$addressSupplement2 and string-length($addressSupplement2)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement2"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:value-of select="$country"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>
            <xsl:if test="$traderIdentificationNumber and string-length($traderIdentificationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tin_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderIdentificationNumber"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$consignorReference and string-length($consignorReference)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'ref_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$consignorReference"/>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="importer">
        <xsl:variable name="name" select="receipt:importer/receipt:name"/>
        <xsl:variable name="street" select="receipt:importer/receipt:street"/>
        <xsl:variable name="country" select="receipt:importer/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:importer/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:importer/receipt:city"/>
        <xsl:variable name="addressSupplement1" select="receipt:importer/receipt:addressSupplement1"/>
        <xsl:variable name="addressSupplement2" select="receipt:importer/receipt:addressSupplement2"/>
        <xsl:variable name="traderIdentificationNumber" select="receipt:importer/receipt:traderIdentificationNumber"/>
        <xsl:variable name="importerReference" select="receipt:importer/receipt:importerReference"/>
        <span class="label">
            <xsl:call-template name="Translate">
                <xsl:with-param name="labelToTranslate" select="'importeur'"/>
            </xsl:call-template>
        </span>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:value-of select="$street"/>
                </td>
            </tr>
            <xsl:if test="$addressSupplement1 and string-length($addressSupplement1)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement1"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$addressSupplement2 and string-length($addressSupplement2)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement2"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:value-of select="$country"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>
            <xsl:if test="$importerReference and string-length($importerReference)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$importerReference"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$traderIdentificationNumber and string-length($traderIdentificationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tin_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderIdentificationNumber"/>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="consignee">
        <xsl:variable name="name" select="receipt:consignee/receipt:name"/>
        <xsl:variable name="street" select="receipt:consignee/receipt:street"/>
        <xsl:variable name="country" select="receipt:consignee/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:consignee/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:consignee/receipt:city"/>
        <xsl:variable name="addressSupplement1" select="receipt:consignee/receipt:addressSupplement1"/>
        <xsl:variable name="addressSupplement2" select="receipt:consignee/receipt:addressSupplement2"/>
        <xsl:variable name="traderIdentificationNumber" select="receipt:consignee/receipt:traderIdentificationNumber"/>
        <xsl:variable name="consigneeReference" select="receipt:consignee/receipt:consigneeReference"/>
        <span class="label">
            <xsl:call-template name="Translate">
                <xsl:with-param name="labelToTranslate" select="'empfaenger'"/>
            </xsl:call-template>
        </span>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:value-of select="$street"/>
                </td>
            </tr>
            <xsl:if test="$addressSupplement1 and string-length($addressSupplement1)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement1"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$addressSupplement2 and string-length($addressSupplement2)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement2"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:value-of select="$country"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>
            <xsl:if test="$traderIdentificationNumber and string-length($traderIdentificationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tin_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderIdentificationNumber"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$consigneeReference and string-length($consigneeReference)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'ref_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$consigneeReference"/>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>
    <xsl:template name="spediteur">
        <!--  Nummer Spedi  -->
        <xsl:variable name="declarantNumber" select="receipt:declarant/receipt:declarantNumber"/>
        <!-- Deklarantennummer -->
        <xsl:variable name="name" select="receipt:declarant/receipt:name"/>
        <!-- Spedi name -->
        <xsl:variable name="street" select="receipt:declarant/receipt:street"/>
        <xsl:variable name="country" select="receipt:declarant/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:declarant/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:declarant/receipt:city"/>
        <xsl:variable name="traderDeclarationNumber" select="receipt:traderDeclarationNumber"/>
        <!-- TIN Spedi -->
        <xsl:variable name="traderIdentificationNumber" select="receipt:declarant/receipt:traderIdentificationNumber"/>
        <!-- Deklarationsnummer Spedi -->
        <xsl:variable name="traderReference" select="receipt:traderReference"/>
        <!--  Dossiernummer Spedi -->
        <span class="label">
            <xsl:call-template name="Translate">
                <xsl:with-param name="labelToTranslate" select="'spediteur'"/>
            </xsl:call-template>
        </span>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>

            <!-- Deklarantennummer -->
            <xsl:if test="$traderDeclarationNumber and string-length($traderDeclarationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'dekl_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$declarantNumber"/>
                    </td>
                </tr>
            </xsl:if>

            <xsl:if test="$street and string-length($street)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$street"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:if test="$country and string-length($country)>0">
                        <xsl:value-of select="$country"/>
                        <xsl:text> </xsl:text>
                    </xsl:if>
                    <xsl:if test="$postalCode and string-length($postalCode)>0">
                        <xsl:value-of select="$postalCode"/>
                    </xsl:if>
                    <xsl:if test="$city and string-length($city)>0">
                        <xsl:text> </xsl:text>
                        <xsl:value-of select="$city"/>
                    </xsl:if>
                </td>
            </tr>

            <!-- Zollanmeldungsnummer-->
            <xsl:if test="$traderDeclarationNumber and string-length($traderDeclarationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'anm_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderDeclarationNumber"/>
                    </td>
                </tr>
            </xsl:if>

            <!-- TIN -->
            <!-- kann nicht null oder leer sein gemaess Schema-Definition; daher keine if-Bedingung -->
            <tr>
                <td>
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'tin_nr'"/>
                    </xsl:call-template>
                    <xsl:value-of select="$traderIdentificationNumber"/>
                </td>
            </tr>
            <!-- Dossiernummer Spediteur -->
            <xsl:if test="$traderReference and string-length($traderReference)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'ref_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderReference"/>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="carrier">
        <xsl:variable name="name" select="receipt:carrier/receipt:name"/>
        <xsl:variable name="street" select="receipt:carrier/receipt:street"/>
        <xsl:variable name="country" select="receipt:carrier/receipt:country"/>
        <xsl:variable name="postalCode" select="receipt:carrier/receipt:postalCode"/>
        <xsl:variable name="city" select="receipt:carrier/receipt:city"/>
        <xsl:variable name="addressSupplement1" select="receipt:carrier/receipt:addressSupplement1"/>
        <xsl:variable name="addressSupplement2" select="receipt:carrier/receipt:addressSupplement2"/>
        <xsl:variable name="traderIdentificationNumber" select="receipt:carrier/receipt:traderIdentificationNumber"/>
        <xsl:variable name="carrierReference" select="receipt:carrier/receipt:carrierReference"/>
        <span class="label">
            <xsl:call-template name="Translate">
                <xsl:with-param name="labelToTranslate" select="'transporteur'"/>
            </xsl:call-template>
        </span>
        <table width="100%" class="content" border="0">
            <tr>
                <td>
                    <xsl:value-of select="$name"/>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:value-of select="$street"/>
                </td>
            </tr>
            <xsl:if test="$addressSupplement1 and string-length($addressSupplement1)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement1"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$addressSupplement2 and string-length($addressSupplement2)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$addressSupplement2"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:value-of select="$country"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$postalCode"/>
                    <xsl:text> </xsl:text>
                    <xsl:value-of select="$city"/>
                </td>
            </tr>

            <!-- TIN -->
            <xsl:if test="$traderIdentificationNumber and string-length($traderIdentificationNumber)>0">
                <tr>
                    <td>
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'tin_nr'"/>
                        </xsl:call-template>
                        <xsl:value-of select="$traderIdentificationNumber"/>
                    </td>
                </tr>
            </xsl:if>

            <xsl:if test="$carrierReference and string-length($carrierReference)>0">
                <tr>
                    <td>
                        <xsl:value-of select="$carrierReference"/>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="headerInformationRight">
        <xsl:variable name="bordereauNumber" select="receipt:bordereauNumber"/>
        <xsl:variable name="deliveryDestination" select="receipt:deliveryDestination"/>
        <xsl:variable name="dispatchCountry" select="receipt:dispatchCountry"/>
        <xsl:variable name="numberOfGoodsItems" select="receipt:numberOfGoodsItems"/>
        <xsl:variable name="VATNumber" select="//receipt:VATNumber"/><!-- Import/Export different schema-->
        <xsl:variable name="transportMode" select="receipt:transportMeans/receipt:transportMode"/>
        <xsl:variable name="transportationType" select="receipt:transportMeans/receipt:transportationType"/>
        <!-- Verkehrszweig -->
        <xsl:variable name="transportationCountry" select="receipt:transportMeans/receipt:transportationCountry"/>
        <xsl:variable name="transportationNumber" select="receipt:transportMeans/receipt:transportationNumber"/>
        <xsl:variable name="container" select="receipt:container"/>
        <xsl:variable name="containerNumber" select="receipt:containerNumber"/>
        <xsl:variable name="previousDocument" select="receipt:previousDocument"/>
        <xsl:variable name="previousDocumentType" select="receipt:previousDocument/receipt:previousDocumentType"/>
        <xsl:variable name="previousDocumentReference" select="receipt:previousDocument/receipt:previousDocumentReference"/>
        <xsl:variable name="additionalInformation" select="receipt:previousDocument/receipt:additionalInformation"/>
        <xsl:variable name="specialMention" select="receipt:specialMention"/>
        <xsl:variable name="reason" select="receipt:reason"/>
        <xsl:variable name="declarationType" select="receipt:declarationType"/>
        <!-- Konto Zoll = customs account-->
        <xsl:variable name="customsAccount" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount"/>
        <xsl:variable name="accountTypeCustoms" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountType"/>
        <xsl:variable name="accountNumberCustoms"
                      select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountNumber"/>
        <xsl:variable name="accountNameCustoms" select="receipt:valuationSummary/receipt:dutyAndAdditionalTaxes/receipt:customsAccount/receipt:accountName"/>
        <!-- Konto MWST = VAT account-->
        <xsl:variable name="VATAccount" select="//receipt:VATAccount"/>
        <xsl:variable name="accountTypeVAT" select="$VATAccount/receipt:accountType"/>
        <xsl:variable name="accountNumberVAT" select="$VATAccount/receipt:accountNumber"/>
        <xsl:variable name="accountNameVAT" select="$VATAccount/receipt:accountName"/>

        <!-- Gesamtbetrag MWST -->
        <xsl:variable name="vatTotalAmount" select="receipt:valuationSummary/receipt:VAT/receipt:totalAmount"/>

        <table width="100%" class="table_0" cellspacing="0">
            <xsl:if test="$bordereauNumber and string-length($bordereauNumber)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'bordereauNumber'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$bordereauNumber"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$deliveryDestination and string-length($deliveryDestination)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'bestimmungsland'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$deliveryDestination"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$dispatchCountry and string-length($dispatchCountry)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'erzeugungsland'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$dispatchCountry"/>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td class="label">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'positionen'"/>
                    </xsl:call-template>
                </td>
                <td class="content">
                    <xsl:value-of select="$numberOfGoodsItems"/>
                </td>
            </tr>
            <!-- Konto Zoll = customs account-->
            <xsl:if test="$customsAccount and string-length($customsAccount)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'konto_zoll'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$accountNumberCustoms"/>
                        <xsl:text>-</xsl:text>
                        <xsl:value-of select="$accountNameCustoms"/>

                    </td>
                </tr>
            </xsl:if>
            <!-- Konto MWST = VAT account-->
            <xsl:if test="$VATAccount and string-length($VATAccount)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'konto_mwst'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$accountNumberVAT"/>
                        <xsl:text>-</xsl:text>
                        <xsl:value-of select="$accountNameVAT"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$VATNumber and string-length($VATNumber)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mwst_nr'"/>
                        </xsl:call-template>
                    </td>
                    <td class="content">
                        <xsl:value-of select="$VATNumber"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$reason and string-length($reason)>0">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'reason'"/>
                        </xsl:call-template>
                        <xsl:text> </xsl:text>
                        <xsl:if test="$declarationType = 2">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'provisorisch'"/>
                            </xsl:call-template>
                        </xsl:if>
                        :
                    </td>
                    <td class="content">
                        <xsl:if test="$reason = 94">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'anderef1j'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 1">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'usnwfldeu'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 10">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'azmozb'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 11">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'voc'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 12">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'chemieprodukte'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 13">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'tarierung'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 14">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'tzs'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 15">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'maschinenanlagen'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 16">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'montage'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 17">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'ozdkd'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 18">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'uritv'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 19">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'agrarprodukte'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 2">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'usnwfldefta'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 20">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'lvm'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 21">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'fnnftaer'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 3">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'usnwfldafha'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 4">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'usnwfel'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 5">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'ursprungsp'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 6">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'tarifierung'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 7">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'uebersiedlungsgut'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 8">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'ausstattungsgut'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 9">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'erbschaftsgut'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 95">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'af30t'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 96">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'af2j'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 97">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'af2m'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 98">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'af6m'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$reason = 99">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'af3j'"/>
                            </xsl:call-template>
                        </xsl:if>
                    </td>
                </tr>
            </xsl:if>
        </table>
        <table width="100%" class="content" border="0" cellspacing="5">
            <tr>
                <td class="label">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'transport'"/>
                    </xsl:call-template>
                </td>
            </tr>
            <tr>
                <td>
                    <xsl:if test="$transportMode and string-length($transportMode)>0">
                        <xsl:if test="$transportMode = 2">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'bahnverkehr'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 3">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'strassenverkehr'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 4">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'luftverkehr'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 5">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'postverkehr'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 7">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'pipelineusw'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 8">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'schiffverkehr'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportMode = 9">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'eigenantrieb'"/>
                            </xsl:call-template>
                        </xsl:if>
                    </xsl:if>
                    <xsl:if test="$transportationType and string-length($transportationType)>0">
                        <xsl:text>, </xsl:text>
                        <xsl:if test="$transportationType = 1">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'pkw'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportationType = 2">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'lkw'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportationType = 3">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'lkwmn'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportationType = 4">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'lkwms'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$transportationType = 99">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'anderetm'"/>
                            </xsl:call-template>
                        </xsl:if>
                    </xsl:if>
                    <xsl:if test="$transportationCountry and string-length($transportationCountry)>0">
                        <xsl:text>, </xsl:text>
                        <xsl:value-of select="$transportationCountry"/>
                    </xsl:if>
                    <xsl:if test="$transportationNumber and string-length($transportationNumber)>0">
                        <xsl:text>, </xsl:text>
                        <xsl:value-of select="$transportationNumber"/>
                    </xsl:if>
                </td>
            </tr>
            <xsl:if test="$container">
                <tr>
                    <td class="label">
                        <div align="left">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'container'"/>
                            </xsl:call-template>
                        </div>
                    </td>
                </tr>
                <xsl:for-each select="$container">
                    <tr>
                        <td>
                            <xsl:value-of select="receipt:containerNumber"/>
                        </td>
                    </tr>
                </xsl:for-each>
            </xsl:if>
            <xsl:if test="$previousDocument">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'vorpapiere'"/>
                        </xsl:call-template>
                    </td>
                </tr>
                <xsl:for-each select="$previousDocument">
                    <tr>
                        <td>
                            <xsl:value-of select="receipt:previousDocumentType"/>
                            <xsl:text>, </xsl:text>
                            <xsl:value-of select="receipt:previousDocumentReference"/>
                            <xsl:text>, </xsl:text>
                            <xsl:value-of select="receipt:additionalInformation"/>
                        </td>
                    </tr>
                </xsl:for-each>
            </xsl:if>
            <xsl:if test="$specialMention">
                <tr>
                    <td class="label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'besondere_vermerke'"/>
                        </xsl:call-template>
                    </td>
                </tr>
                <xsl:for-each select="$specialMention">
                    <tr>
                        <td>
                            <xsl:value-of select="receipt:text"/>
                        </td>
                    </tr>
                </xsl:for-each>
            </xsl:if>

            <xsl:if test="$vatTotalAmount and string-length($vatTotalAmount)>0">
                <tr>
                    <td class="label">
                        <font size="4">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'gesamtbetrag_mwst'"/>
                            </xsl:call-template>
                        </font>
                    </td>
                    <td>
                        <font size="4">
                            <xsl:value-of select="$vatTotalAmount"/>
                        </font>
                    </td>
                </tr>
            </xsl:if>


            <!-- valuationSummary refundCustomsDuties -->
            <xsl:if test="($documentType = 'taxationDecisionCustomsDuties')">
                <tr>
                    <table width="200" class="table_1">
                        <xsl:call-template name="valuationSummaryCustomDuties"/>
                    </table>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>

    <xsl:template name="legalAdvisory">
        <xsl:variable name="title" select="receipt:title"/>
        <xsl:variable name="text" select="receipt:text"/>
        <tr>
            <td class="label">

                <xsl:value-of select="$title"/>
            </td>
        </tr>
        <tr>
            <td class="content">
                <xsl:value-of select="$text"/>
            </td>
        </tr>
        <br/>

    </xsl:template>

    <!-- GoodsItems -->
    <xsl:template name="goodsItem">
        <xsl:variable name="customsItemNumber" select="receipt:customsItemNumber"/>
        <xsl:variable name="description" select="receipt:description"/>
        <xsl:variable name="statisticalCode" select="receipt:statisticalCode"/>
        <xsl:variable name="commodityCode" select="receipt:commodityCode"/>
        <xsl:variable name="customsClearanceType" select="receipt:statistic/receipt:customsClearanceType"/>
        <xsl:variable name="refund" select="receipt:refund"/>
        <xsl:variable name="refundType" select="receipt:refund/receipt:refundType"/>
        <!-- Eigenmasse -->
        <xsl:variable name="netMass" select="receipt:netMass"/>
        <!-- Rohmasse -->
        <xsl:variable name="grossMass" select="receipt:grossMass"/>
        <!-- Stat. Wert -->
        <xsl:variable name="statisticalValue" select="receipt:statistic/receipt:statisticalValue"/>
        <!-- Zollnetogewicht -->
        <xsl:variable name="customsNetWeight" select="receipt:customsNetWeight"/>
        <xsl:variable name="specialMention" select="receipt:specialMention"/>
        <xsl:variable name="additionalUnit" select="receipt:additionalUnit"/>
        <xsl:variable name="VOCQuantity" select="receipt:refund/receipt:VOCQuantity"/>
        <xsl:variable name="packaging" select="receipt:packaging"/>
        <xsl:variable name="permit" select="receipt:permit"/>
        <xsl:variable name="producedDocument" select="receipt:producedDocument"/>
        <xsl:variable name="processType" select="receipt:repairAndRefinement/receipt:processType"/>
        <xsl:variable name="originPreference" select="receipt:origin/receipt:preference"/>
        <xsl:variable name="originCountry" select="receipt:origin/receipt:originCountry"/>
        <!--  Stammdaten -->
        <xsl:variable name="customsClearanceTypeText">
            <xsl:choose>
                <xsl:when test="$customsClearanceType = '1'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'normalabfertigung'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '2'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'veredelungsverkehr'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '3'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'ausbesserungsverkehr'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '4'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckwaren'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '5'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'zollerleichterung'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '6'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'tabak'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '7'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'zollfreie_abfertigung_gem_zg'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '8'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'abgabenfreie_abfertigung'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '9'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'sonderfaelle_ex_zil_abfertigungscode_9'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '10'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckgemaesszollrecht'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$customsClearanceType = '11'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckgemaesszollrechtmwstrecht'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$customsClearanceType"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="refundTypeText">
            <xsl:choose>
                <xsl:when test="$refundType = '1'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckerstattung_voc'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$refundType = '2'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'antrag_ausfuhrbeitraeger_lw'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$refundType = '3'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'antrag_auf_alkohol'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$refundType = '4'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'antrag_rueckerstattungstyp_1_bis_3'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$refundType = '5'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckerstattung_biersteuer'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$refundType = '6'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'rueckerstattung_andere'"/>
                    </xsl:call-template>
                </xsl:when>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="processTypeText">
            <xsl:choose>
                <xsl:when test="$processType = '1'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'ordentliches_verfahren'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$processType = '2'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'vereinfachtes_verfahren'"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:when test="$processType = '3'">
                    <xsl:call-template name="Translate">
                        <xsl:with-param name="labelToTranslate" select="'besonderes_verfahren'"/>
                    </xsl:call-template>
                </xsl:when>
            </xsl:choose>
        </xsl:variable>
        <table width="788" border="0" class="content">
            <tr>
                <hr/>
            </tr>
            <tr>
                <td colspan="2">
                    <strong>
                        <xsl:value-of select="receipt:customsItemNumber"/>
                        <xsl:text>   </xsl:text>
                        <xsl:value-of select="receipt:description"/>
                    </strong>
                </td>
                <td width="230" align="right">
                    <strong>
                        <xsl:value-of select="$commodityCode"/>
                        <xsl:if test="$statisticalCode">
                            <xsl:text>   </xsl:text>
                            <xsl:value-of select="$statisticalCode"/>
                        </xsl:if>
                    </strong>
                </td>
            </tr>
            <xsl:if test="$customsClearanceType or $refund">
                <tr>
                    <td>
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label" width="50%">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'veranlagungstyp'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$customsClearanceTypeText"/>
                                </td>
                            </tr>
                        </table>
                    </td>
                    <td>
                        <xsl:if test="$refundTypeText != ''">
                            <span class="label">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'rueckerstattungstyp'"/>
                                </xsl:call-template>
                            </span>
                            <xsl:value-of select="$refundTypeText"/>
                        </xsl:if>
                    </td>
                    <td>
                        <xsl:if test="$originPreference">
                            <span class="label">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'praeferenz'"/>
                                </xsl:call-template>
                            </span>
                        </xsl:if>
                        <xsl:if test="$originPreference = 1">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'praeferenz1'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$originPreference = 0">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'praeferenz0'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="$originCountry">
                            (<xsl:value-of select="$originCountry"/>)
                        </xsl:if>
                    </td>

                </tr>
            </xsl:if>
            <!--- ValuationDetails -->
            <xsl:choose>
                <xsl:when test="$documentType = 'taxationDecisionVAT'">
                    <tr>
                        <xsl:call-template name="valuationDetailsTaxationDecisionVAT"/>
                    </tr>
                </xsl:when>
                <xsl:when test="$documentType = 'taxationDecisionCustomsDuties'">
                    <tr>
                        <xsl:call-template name="valuationDetailsTaxationDecisionCustomsDuties"/>
                    </tr>
                </xsl:when>
                <!-- No details for Export -->
            </xsl:choose>

            <!-- more GoodsItems -->

            <xsl:if test="$processType">
                <tr>
                    <td>
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label" width="12%">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'verfahren'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$processTypeText"/>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td>
                    <xsl:if test="$netMass">
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label" width="50%">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'eigenmasse'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$netMass"/>
                                </td>
                            </tr>
                        </table>
                    </xsl:if>
                </td>
                <td>
                    <xsl:if test="$grossMass and ($documentType = 'taxationDecisionExport')">
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'rohmasse'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$grossMass"/>
                    </xsl:if>
                </td>
                <td>
                    <xsl:if test="$statisticalValue and ($documentType = 'taxationDecisionExport')">
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'statwert'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$statisticalValue"/>
                    </xsl:if>
                </td>
            </tr>
            <tr>
                <td>
                    <table width="100%" border="0" class="content">
                        <xsl:if test="$customsNetWeight>0">
                            <tr>
                                <td class="label" width="50%">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'nettogewicht'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$customsNetWeight"/>
                                </td>
                            </tr>
                        </xsl:if>
                        <xsl:if test="$additionalUnit>0">
                            <tr>
                                <td class="label" width="50%">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'zusatzmenge'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:value-of select="$additionalUnit"/>
                                </td>
                            </tr>
                        </xsl:if>
                    </table>
                </td>
                <td></td>
                <td valign="top" colspan="2">
                    <xsl:if test="$VOCQuantity and string-length($VOCQuantity)>0">
                        <span class="label">
                            <xsl:call-template name="Translate">
                                <xsl:with-param name="labelToTranslate" select="'vocmenge'"/>
                            </xsl:call-template>
                        </span>
                        <xsl:value-of select="$VOCQuantity"/>
                    </xsl:if>
                </td>
            </tr>
            <xsl:if test="$specialMention">
                <tr>
                    <td>
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'besondere_vermerke'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:for-each select="$specialMention/receipt:text">
                                <tr>
                                    <td>
                                        <xsl:value-of select="."/>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <tr>
                <td valign="top">
                    <table width="100%" border="0" class="content">
                        <tr>
                            <td class="label">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'packstuecke'"/>
                                </xsl:call-template>
                            </td>
                        </tr>
                        <tr>
                            <td class="label">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'anzahl_art_nummer'"/>
                                </xsl:call-template>
                            </td>
                        </tr>
                        <xsl:for-each select="$packaging">
                            <tr>
                                <td>
                                    <xsl:value-of select="./receipt:packagingType"/>
                                    <xsl:text>, </xsl:text>
                                    <xsl:value-of select="./receipt:quantity"/>
                                    <xsl:text>, </xsl:text>
                                    <xsl:value-of select="./receipt:packagingReferenceNumber"/>
                                </td>
                            </tr>
                        </xsl:for-each>
                    </table>
                </td>
                <xsl:if test="$permit">
                    <td valign="top">
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'bewilligungen'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'typ_stelle_etc'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:for-each select="$permit">
                                <tr>
                                    <td>
                                        <xsl:value-of select="./receipt:permitAuthority"/>
                                        <xsl:text>, </xsl:text>
                                        <xsl:value-of select="./receipt:permitNumber"/>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                    </td>
                </xsl:if>
                <xsl:if test="$producedDocument">
                    <td valign="top">
                        <table width="100%" border="0" class="content">
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'unterlagen'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <tr>
                                <td class="label">
                                    <xsl:call-template name="Translate">
                                        <xsl:with-param name="labelToTranslate" select="'art_nummer_datum'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:for-each select="$producedDocument">
                                <tr>
                                    <td>
                                        <xsl:value-of select="./receipt:documentType"/>
                                        <xsl:text>, </xsl:text>
                                        <xsl:value-of select="./receipt:documentReferenceNumber"/>
                                        <xsl:text>, </xsl:text>
                                        <xsl:value-of select="./receipt:issueDate"/>
                                        <xsl:text>, </xsl:text>
                                        <xsl:value-of select="./receipt:additionalInformation"/>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                    </td>
                </xsl:if>
            </tr>
        </table>
        <br/>
    </xsl:template>

    <xsl:template name="valuationDetailsTaxationDecisionVAT">
        <xsl:variable name="amount" select="receipt:valuationDetails/receipt:VAT/receipt:amount"/>
        <xsl:variable name="duties" select="receipt:valuationDetails/receipt:VAT/receipt:duties"/>
        <xsl:variable name="basisForAssessment" select="receipt:valuationDetails/receipt:VAT/receipt:basisForAssessment"/>
        <xsl:variable name="rate" select="receipt:valuationDetails/receipt:VAT/receipt:rate"/>
        <xsl:variable name="comment" select="receipt:valuationDetails/receipt:VAT/receipt:comment"/>
        <xsl:variable name="VATValue" select="receipt:valuationDetails/receipt:VAT/receipt:VATValue"/>
        <xsl:variable name="VATNumber" select="receipt:valuationDetails/receipt:VAT/receipt:VATNumber"/>

        <td colspan="3">
            <table width="100%" class="table_0">
                <tr>
                    <td/>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'betrag'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'ansatz_prozent'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mwst'"/>
                        </xsl:call-template>
                    </td>
                </tr>
                <tr>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'mwst_wert'"/>
                        </xsl:call-template>
                    </td>
                    <td align="right" class="td_content">
                        <xsl:value-of select="$VATValue"/>
                    </td>
                    <td align="right" bgcolor="#C0C0C0" class="td_content">
                        <xsl:text/>
                    </td>
                    <td align="right" bgcolor="#C0C0C0" class="td_content">
                        <xsl:text/>
                    </td>
                </tr>
                <tr>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'abgaben'"/>
                        </xsl:call-template>
                    </td>
                    <td align="right" class="td_content">
                        <xsl:value-of select="$duties"/>
                    </td>
                    <td align="right" bgcolor="#C0C0C0" class="td_content">
                        <xsl:text/>
                    </td>
                    <td align="right" bgcolor="#C0C0C0" class="td_content">
                        <xsl:text/>
                    </td>
                </tr>
                <tr>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'bemessungsgrundlage'"/>
                        </xsl:call-template>
                    </td>
                    <td align="right" class="td_content">
                        <b>
                            <xsl:value-of select="$basisForAssessment"/>
                        </b>
                    </td>
                    <td align="right" class="td_content">
                        <b>
                            <xsl:value-of select="$rate"/>
                        </b>
                    </td>
                    <td align="right" class="td_content">
                        <b>
                            <xsl:value-of select="$amount"/>
                        </b>
                    </td>
                </tr>
            </table>
        </td>
    </xsl:template>

    <xsl:template name="valuationDetailsTaxationDecisionCustomsDuties">
        <xsl:variable name="totalAmount" select="receipt:valuationDetails/receipt:dutyAndAdditionalTaxes/receipt:totalAmount"/>
        <td colspan="3">
            <table width="100%" class="table_0">
                <tr>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'einnahmenart'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'bemessungsgrundlage_zoll'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'volumenprozent'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'ansatz'"/>
                        </xsl:call-template>
                    </td>
                    <td class="td_label">
                        <xsl:call-template name="Translate">
                            <xsl:with-param name="labelToTranslate" select="'betrag'"/>
                        </xsl:call-template>
                    </td>
                </tr>

                <xsl:for-each select="receipt:valuationDetails/receipt:dutyAndAdditionalTaxes/receipt:detail">
                    <xsl:variable name="type" select="receipt:type"/>
                    <xsl:variable name="basisForAssessment" select="receipt:basisForAssessment"/>
                    <xsl:variable name="rate" select="receipt:rate"/>
                    <xsl:variable name="alcoholLevel" select="receipt:alcoholLevel"/>
                    <xsl:variable name="amount" select="receipt:amount"/>
                    <!-- Stammdaten AdditionalTaxes -->
                    <xsl:variable name="additionalTaxesTypeText">
                        <xsl:choose>
                            <xsl:when test="$type = '150'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'andereGebuehren'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '151'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'edelmetallkontrol'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '280'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'monopolgebuehren'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '290'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'grenztieraerztliche_untersuchung'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '291'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'kanzleigebuehr'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '292'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'fauna-gebuehr'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '430'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'provuebersiedlung'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '450'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'tabaksteuer'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '460'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'banderolensteuer'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '465'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'SOTA-gebuehren'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '470'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'tabakpraeventionsfonds'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '480'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'biersteuer'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '600'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteue_benzin'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '601'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_diesel'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '602'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_petrol'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '603'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_anderen_treibstoffen'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '605'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'minoestflugbenzin'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '606'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'minoestflugpetrol'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '620'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_benzin'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '621'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_diesel'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '622'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_petrol'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '623'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuerzuschlag_anderen_treibstoffen'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '625'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'minoestzusflugbenz'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '626'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'minoestzusflugpetrol'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '640'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'mineraloelsteuer_mineraloelprodukten'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '660'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'automobilsteuer'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '700'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_VOC'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '710'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'lenkungsabgaben_heizoel'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '720'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_autobenzin'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '730'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'lenkungsabgabe_dieseloel'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '740'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_heizoel'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '741'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_erdgas'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '742'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_mineraloelen'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '743'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'CO2-Abgabe_kohle'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '790'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'zollanteil_pflanzenschutzabgabe'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '791'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'pflanzschutzgeb'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '970'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'vorgezogene_entsorgungsgebuehr'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '110'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'zollabgaben'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="$type = '995'">
                                <xsl:call-template name="Translate">
                                    <xsl:with-param name="labelToTranslate" select="'zollbeguenstkontgeb'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$type"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>


                    <tr>
                        <!-- Einnahmeart -->
                        <td align="left" class="td_content">
                            <xsl:value-of select="$additionalTaxesTypeText"/>
                        </td>
                        <!-- optional -->
                        <td align="left" class="td_content">
                            <xsl:value-of select="$basisForAssessment"/>
                            <xsl:if test="string-length($basisForAssessment) = 0">
                                <xsl:text>-</xsl:text>
                            </xsl:if>
                        </td>
                        <!-- optional -->
                        <td align="right" class="td_content">
                            <xsl:value-of select="$alcoholLevel"/>
                            <xsl:if test="string-length($alcoholLevel) = 0">
                                <xsl:text>-</xsl:text>
                            </xsl:if>
                        </td>
                        <!-- optional -->
                        <td align="left" class="td_content">
                            <xsl:value-of select="$rate"/>
                            <xsl:if test="string-length($rate) = 0">
                                <xsl:text>-</xsl:text>
                            </xsl:if>
                        </td>
                        <!-- optional -->
                        <td align="right" class="td_content">
                            <xsl:value-of select="$amount"/>
                        </td>
                    </tr>
                </xsl:for-each>
                <tr>
                    <td/>
                    <td/>
                    <td/>
                    <td/>
                    <td align="right" class="td_content">
                        <b>
                            <xsl:value-of select="$totalAmount"/>
                        </b>
                    </td>
                </tr>
            </table>
        </td>
    </xsl:template>

    <!-- TODO: abkl&#228;ren ob noch gebraucht wird. Biser nie Feedback bekommen. Sonst dieses Template löschen -->
    <xsl:template match="receipt:receiptRequestResponse/receipt:receiptList" name="receiptList">
        <xsl:variable name="requestDate" select="receipt:receiptRequestSummary/receipt:requestDate"/>
        <xsl:variable name="requestTime" select="receipt:receiptRequestSummary/receipt:requestTime"/>
        <xsl:variable name="numberOfDocumentsFound" select="receipt:receiptRequestSummary/receipt:numberOfDocumentsFound"/>
        <html>
            <head>
                <title>Liste der Zollanmeldungsnummern</title>
                <style type="text/
">
                    <xsl:text>
					.title {font-family:Arial, Helvetica, sans-serif; font-size: 20px; font-style:bold}
					.label {font-family:Arial, Helvetica, sans-serif;font-size: 12px; font-style:italic}
					.content {font-family:Arial, Helvetica, sans-serif;font-size: 12px}
					.header {font-family:Arial, Helvetica, sans-serif;font-size: 10px}
					</xsl:text>
                </style>
            </head>
            <body>
                <table width="787" border="0">
                    <tr>
                        <td width="100%" scope="row">
                            <xsl:call-template name="header"/>
                        </td>
                    </tr>
                    <tr>
                        <td scope="row">
                            <div align="left">
                                <span class="title">Liste der Zollanmeldungsnummern</span>
                            </div>
                            <p>
                            </p>
                            <p>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <td scope="row">
                            <div align="left">
                                <table width="100%" class="content" border="0">
                                    <tr>
                                        <td scope="row" class="label" width="30%">
                                            <xsl:text>Abfragedatum:</xsl:text>
                                            <xsl:value-of select="$requestDate"/>
                                            <xsl:text>, </xsl:text>
                                            <xsl:value-of select="$requestTime"/>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td scope="row" class="label" width="30%">
                                            <xsl:text>Anzahl gefundener Veranlagungsverf&#252;gungen:</xsl:text>
                                            <xsl:value-of select="$numberOfDocumentsFound"/>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </td>
                    </tr>
                </table>
                <p>
                </p>
                <p>
                </p>
                <p>
                </p>
                <p>
                </p>
                <xsl:for-each select="receipt:receiptInformation">
                    <xsl:call-template name="receiptInformation"/>
                </xsl:for-each>
            </body>
        </html>
    </xsl:template>

    <xsl:template name="receiptInformation">
        <xsl:variable name="documentType" select="receipt:documentType"/>
        <xsl:variable name="customsDeclarationNumber" select="receipt:customsDeclarationNumber"/>
        <xsl:variable name="customsDeclarationVersion" select="receipt:customsDeclarationVersion"/>
        <xsl:variable name="documentDate" select="receipt:documentDate"/>
        <xsl:variable name="documentTime" select="receipt:documentTime"/>
        <xsl:variable name="documentStatus" select="receipt:documentStatus"/>
        <xsl:variable name="consigneeName" select="receipt:consigneeName"/>
        <xsl:variable name="transportationNumber" select="receipt:transportationNumber"/>
        <xsl:variable name="customsOfficeNumber" select="receipt:customsOfficeNumber"/>
        <table width="788" border="0" class="content">
            <tr>
                <hr>
                </hr>
            </tr>
            <tr>
                <td colspan="2">
                    <strong>
                        <xsl:value-of select="$customsDeclarationNumber"/>
                    </strong>
                    .
                    <strong>
                        <xsl:value-of select="$customsDeclarationVersion"/>
                    </strong>
                </td>
            </tr>
            <tr>
                <td width="60%">
                    <span class="label">Empf&#228;ngername:</span>
                    <xsl:value-of select="$consigneeName"/>
                </td>
                <td>
                    <table width="100%" border="0" class="content">
                        <tr>
                            <td class="label">Ausstellungsdatum:
                                <xsl:value-of select="$documentDate"/>
                                <xsl:text>, </xsl:text>
                                <xsl:value-of select="$documentTime"/>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
            <tr>
                <td>
                    <span class="label">Dokumenttyp:</span>
                    <xsl:value-of select="$documentType"/>
                </td>
                <td>
                    <span class="label">Zollstellennummer:</span>
                    <xsl:value-of select="$customsOfficeNumber"/>
                </td>
            </tr>
            <tr>
                <td>
                    <span class="label">Status:</span>
                    <xsl:value-of select="$documentStatus"/>
                </td>
                <td>
                    <span class="label">TransportNummer:</span>
                    <xsl:if test="$transportationNumber and string-length($transportationNumber)>0">
                        <xsl:value-of select="$transportationNumber"/>
                    </xsl:if>
                </td>
            </tr>
        </table>
        <br>
        </br>
    </xsl:template>
    <!-- END TODO: abkl&#228;ren ob noch gebraucht wird. Biser nie Feedback bekommen. Sonst dieses Template löschen -->

    <!-- Translations DE, FR, IT -->
    <xsl:template name="Translate">
        <xsl:param name="labelToTranslate" select="/.."/>
        <xsl:variable name="lower">abcdefghijklmnopqrstuvwxyz</xsl:variable>
        <xsl:variable name="upper">ABCDEFGHIJKLMNOPQRSTUVWXYZ</xsl:variable>
        <xsl:choose>
            <xsl:when test="translate($documentLanguage,$lower,$upper) = 'IT'">
                <xsl:call-template name="TranslateIT">
                    <xsl:with-param name="label" select="$labelToTranslate"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="translate($documentLanguage,$lower,$upper) = 'FR'">
                <xsl:call-template name="TranslateFR">
                    <xsl:with-param name="label" select="$labelToTranslate"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="translate($documentLanguage,$lower,$upper) = 'DE'">
                <xsl:call-template name="TranslateDE">
                    <xsl:with-param name="label" select="$labelToTranslate"/>
                </xsl:call-template>
            </xsl:when>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="TranslateFR">
        <xsl:param name="label" select="/.."/>
        <xsl:choose>
            <xsl:when test="$label = 'art_nummer_datum'">
                <xsl:text>(genre, numéro, date, données supplémentaires)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'unterlagen'">
                <xsl:text>Documents:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'typ_stelle_etc'">
                <xsl:text>(office, numéro)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bewilligungen'">
                <xsl:text>Permis:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anzahl_art_nummer'">
                <xsl:text>(genre, nombre, numéro)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'packstuecke'">
                <xsl:text>Colis:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vocmenge'">
                <xsl:text>Quantité COV:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Mentions spéciales:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenmasse'">
                <xsl:text>Masse effective:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rohmasse'">
                <xsl:text>Masse brute:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'statwert'">
                <xsl:text>Valeur stat.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattungstyp'">
                <xsl:text>Genre de remboursement:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungstyp'">
                <xsl:text>Genre de taxation:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_auf_alkohol'">
                <xsl:text>Avec demande pour l'alcool</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_rueckerstattungstyp_1_bis_3'">
                <xsl:text>Avec demande pour aux moins 2 des codes 1 &#224; 3</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_biersteuer'">
                <xsl:text>Remboursement de l'alcool sur la bière</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_andere'">
                <xsl:text>autres remboursements</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_ausfuhrbeitraeger_lw'">
                <xsl:text>Avec demande de contributions a l'exportation pour produits agricoles de base</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_voc'">
                <xsl:text>remboursement de la taxe d' indication sur les COV</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sonderfaelle_ex_zil_abfertigungscode_9:'">
                <xsl:text>cas spéciaux ex SDI code de dédouanement 9</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrecht'">
                <xsl:text>marchandises en retour selon le droit douanier</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrechtmwstrecht'">
                <xsl:text>marchandises en retour selon le droit douanier/TVA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgabenfreie_abfertigung'">
                <xsl:text>Dédouanement en franchise de redevances</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollfreie_abfertigung_gem_zg'">
                <xsl:text>Dédouanement en franchise de droits de douane selon LD</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabak'">
                <xsl:text>Tabac</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollerleichterung'">
                <xsl:text>Allegement douanier</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckwaren'">
                <xsl:text>marchandises en retour</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausbesserungsverkehr'">
                <xsl:text>Trafic de réparation</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veredelungsverkehr'">
                <xsl:text>trafic de perfectionnement</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'normalabfertigung'">
                <xsl:text>Dédouanement normal</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Mentions spéciales</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorpapiere'">
                <xsl:text>Documents précédents:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'container'">
                <xsl:text>Conteneur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transport'">
                <xsl:text>Transport:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_nr'">
                <xsl:text>No TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'positionen'">
                <xsl:text>positions:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transporteur'">
                <xsl:text>Transporteur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dekl_nr'">
                <xsl:text>Dekl.Nr.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'spediteur'">
                <xsl:text>Transitaire:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'empfaenger'">
                <xsl:text>Destinataire:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ref_nr'">
                <xsl:text>Ref.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tin_nr'">
                <xsl:text>No de transitaire/TIN/UID:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'versender'">
                <xsl:text>Nom de l'expéditeur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessische_zollverwaltung_ezv'">
                <xsl:text>Administration fédérale des douanes AFD</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessisches_finanzdepartement_efd'">
                <xsl:text>Département fédéral des finances DFF</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rechtsmittelbelehrung'">
                <xsl:text>Indication des voies de droit:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstellungsdatum'">
                <xsl:text>Date d'etablissement:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annahmedatum'">
                <xsl:text>Date d'acceptation:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung_ausfuhr'">
                <xsl:text>DECISION DE TAXATION A L'EXPORTATION</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung'">
                <xsl:text>Decision de Taxation</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ordentliches_verfahren'">
                <xsl:text>procédure ordinaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vereinfachtes_verfahren'">
                <xsl:text>procédure simplifiée</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besonderes_verfahren'">
                <xsl:text>Procédure spéciale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annullation'">
                <xsl:text>Annulation</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'verfahren'">
                <xsl:text>procédure</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'importeur'">
                <xsl:text>Importateur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'reason'">
                <xsl:text>Motif:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bestimmungsland'">
                <xsl:text>Pays de destination:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erzeugungsland'">
                <xsl:text>Pays de production:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bordereauNumber'">
                <xsl:text>numéro du bordereau</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'VATAccount'">
                <xsl:text>Compte TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto_zoll'">
                <xsl:text>Compte douane:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst'">
                <xsl:text>TVA [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mehrwertsteuernummer'">
                <xsl:text>numéro de TVA en cas de remboursement</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgaben'">
                <xsl:text>redevances selon décision de taxation douane:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz_prozent'">
                <xsl:text>Taux [%]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Montant [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kommentar_zu_MWST'">
                <xsl:text>commentaire TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_wert'">
                <xsl:text>Valeur TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_rueck'">
                <xsl:text>TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmenart'">
                <xsl:text>genre de recette:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage'">
                <xsl:text>base de calcul TVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage_zoll'">
                <xsl:text>Base de calcul:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz'">
                <xsl:text>Taux [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'volumenprozent'">
                <xsl:text>Vol-%:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Montant [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag'">
                <xsl:text>montant total [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_ohne'">
                <xsl:text>Montant total:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zusatzangabenschluessel'">
                <xsl:text>clé de redevances supplémentaires:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettogewicht'">
                <xsl:text>Poids net:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sachbearbeiter'">
                <xsl:text>Contact:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'referenz_zoll'">
                <xsl:text>référence douane:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'berichtigungsgrund'">
                <xsl:text>Motif de la rectification:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dossiernummer_spediteur'">
                <xsl:text>numéro de dossier transitaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kunde'">
                <xsl:text>Client:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto'">
                <xsl:text>Compte:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmeart'">
                <xsl:text>Genre de perception:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettorueckverguetung'">
                <xsl:text>Remboursement net:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anm_nr'">
                <xsl:text>numéro de déclaration:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'monopolgebuehren'">
                <xsl:text>Droits de monopole</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'grenztieraerztliche_untersuchung'">
                <xsl:text>Visite vétérinaire de frontière / taxe de visite</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fauna-gebuehr'">
                <xsl:text>Emoluments CITES Fauna (contrôle de la conservation des espèces)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabaksteuer'">
                <xsl:text>Impôt sur le tabac</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'banderolensteuer'">
                <xsl:text>Impôt sur les banderoles</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'SOTA-gebuehren'">
                <xsl:text>Redevances SOTA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabakpraeventionsfonds'">
                <xsl:text>Fonds de prévention du tabagisme</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'biersteuer'">
                <xsl:text>Impôt sur la bière</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteue_benzin'">
                <xsl:text>Impôts sur les huiles minérales grevant le carburant essence</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_diesel'">
                <xsl:text>Impôt sur les huiles minérales grevant le carburant Diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_petrol'">
                <xsl:text>Impôt sur les huiles minérales grevant le carburant petrole</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_anderen_treibstoffen'">
                <xsl:text>Impôt sur les huiles minérales grevant les autres carburants</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_benzin'">
                <xsl:text>Surtaxe sur les huiles minérales grevant le carburant essence</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_diesel'">
                <xsl:text>Surtaxe sur les huiles minérales grevant le carburant Diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_petrol'">
                <xsl:text>Surtaxe sur les huiles minérales grevant le carburant pétrole</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_anderen_treibstoffen'">
                <xsl:text>Surtaxe sur les huiles minérales grevant les autres carburants</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_mineraloelprodukten'">
                <xsl:text>Impôt sur les huiles minérales grevant les combustibles et autres produits &#224; base d'huiles minérales</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'automobilsteuer'">
                <xsl:text>Impôt sur les automobiles</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_VOC'">
                <xsl:text>Taxe d'incitation sur les composés organiques volatils (COV)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgaben_heizoel'">
                <xsl:text>Taxe d'incitation sur l'huile de chauffage extra légère</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_autobenzin'">
                <xsl:text>Taxe d'incitation sur l'essence</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_dieseloel'">
                <xsl:text>Taxe d'incitation sur l'huile diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_heizoel'">
                <xsl:text>Taxe sur le CO2 sur l'huile de chauffage</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_erdgas'">
                <xsl:text>Taxe sur le CO2 sur le gaz naturel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_mineraloelen'">
                <xsl:text>Taxe sur le CO2 sur les autres huiles minérales</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_kohle'">
                <xsl:text>Taxe sur le CO2 sur le charbon/coke de pétrole</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollanteil_pflanzenschutzabgabe'">
                <xsl:text>Quote-part douanière, taxe phytosanitaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorgezogene_entsorgungsgebuehr'">
                <xsl:text>Taxe d'élimination anticipée sur les bouteilles en verre pour boissons</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollabgaben'">
                <xsl:text>droits de douane</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch'">
                <xsl:text>provisoire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch_with'">
                <xsl:text>provisoire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_mwst'">
                <xsl:text>Montant total TVA [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz'">
                <xsl:text>Préférence</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'andereGebuehren'">
                <xsl:text>Autres taxes</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'edelmetallkontrol'">
                <xsl:text>Taxes du contrôle des métaux précieux</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kanzleigebuehr'">
                <xsl:text>Emoluments de chancellerie pour le vétérinaire de frontière</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provuebersiedlung'">
                <xsl:text>Prov. Effets de déménagement</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugbenzin'">
                <xsl:text>Impôts sur les huiles minérales grevant le carburant essence pour avions</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugpetrol'">
                <xsl:text>Impôt sur les huiles minérales grevant le carburant petrole pour avions</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugbenz'">
                <xsl:text>Surtaxe sur les huiles minérales grevant le carburant essence pour avions</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pkw'">
                <xsl:text>Voiture</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkw'">
                <xsl:text>Camion</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwmn'">
                <xsl:text>Camion avec remorque normale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwms'">
                <xsl:text>Camion avec semi-remorque</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderetm'">
                <xsl:text>Autres</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bahnverkehr'">
                <xsl:text>Trafic ferroviaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'strassenverkehr'">
                <xsl:text>Trafic routier</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'luftverkehr'">
                <xsl:text>Trafic aérien</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'postverkehr'">
                <xsl:text>Trafic postal</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pipelineusw'">
                <xsl:text>Pipeline, etc.</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'schiffverkehr'">
                <xsl:text>Trafic par eau</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenantrieb'">
                <xsl:text>Autopropulsion</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderef1j'">
                <xsl:text>Autres; Délai 1 an</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldeu'">
                <xsl:text>Preuve d'origine pour les pays de l'UE</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'azmozb'">
                <xsl:text>Autres réductions des droits de douane ou d'exonérations des droit de douane</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'voc'">
                <xsl:text>COV</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'chemieprodukte'">
                <xsl:text>Produits chimiques</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarierung'">
                <xsl:text>Tarage</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tzs'">
                <xsl:text>Animaux de boucherie+D122</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'maschinenanlagen'">
                <xsl:text>Installations de machines</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'montage'">
                <xsl:text>Montage/frais de montage</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ozdkd'">
                <xsl:text>Autorisation DGD/DA pour l'admission temporaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uritv'">
                <xsl:text>Irrégularités dans le régime du transit / de l'admission temporaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'agrarprodukte'">
                <xsl:text>Produits agricoles</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldefta'">
                <xsl:text>Preuve d'origine pour les pays de l'AELE</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lvm'">
                <xsl:text>Entreposage de marchandises de grande consommation</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fnnftaer'">
                <xsl:text>Manque le numéro de preuve pour carburants issus de matières premières renouvelables</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldafha'">
                <xsl:text>Preuve d'origine pour les pays de l'aALE</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfel'">
                <xsl:text>Preuve d'origine pour les pays en voie de développement</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ursprungsp'">
                <xsl:text>Contrôle de l'origine </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarifierung'">
                <xsl:text>Tarification</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uebersiedlungsgut'">
                <xsl:text>Effets de déménagement</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstattungsgut'">
                <xsl:text>Trousseaux de mariage</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erbschaftsgut'">
                <xsl:text>Effets de succession</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af30t'">
                <xsl:text>Autres; Délai 30 jours</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2j'">
                <xsl:text>Autres; Délai 2 ans</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2m'">
                <xsl:text>Autres; Délai 2 mois</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af6m'">
                <xsl:text>Autres; Délai 6 mois </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af3j'">
                <xsl:text>Autres; Délai 3 ans</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zusatzmenge'">
                <xsl:text>Quantité suppl.</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz1'">
                <xsl:text>Oui</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz0'">
                <xsl:text>Non</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'definitiv'">
                <xsl:text>définitif</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'space'">
                <xsl:text> </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'import'">
                <xsl:text>Import</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'export'">
                <xsl:text>Export</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugpetrol'">
                <xsl:text>Surtaxe sur les huiles minérales grevant le carburant pétrole pour avions</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pflanzschutzgeb'">
                <xsl:text>taxe phytosanitaire</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollbeguenstkontgeb'">
                <xsl:text>Allègement douanier, taxe de contrôle</xsl:text>
            </xsl:when>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="TranslateDE">
        <xsl:param name="label" select="/.."/>
        <xsl:choose>
            <xsl:when test="$label = 'art_nummer_datum'">
                <xsl:text>(Art,Nummer,Datum,zus&#228;tzliche Angaben)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'unterlagen'">
                <xsl:text>Unterlagen:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'typ_stelle_etc'">
                <xsl:text>(Stelle,Nummer)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bewilligungen'">
                <xsl:text>Bewilligungen:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anzahl_art_nummer'">
                <xsl:text>(Art,Anzahl,Nummer)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'packstuecke'">
                <xsl:text>Packst&#252;cke:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vocmenge'">
                <xsl:text>VOC Menge:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Besondere Vermerke:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenmasse'">
                <xsl:text>Eigenmasse:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rohmasse'">
                <xsl:text>Rohmasse:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'statwert'">
                <xsl:text>Stat. Wert:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattungstyp'">
                <xsl:text>R&#252;ckerstattungstyp:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungstyp'">
                <xsl:text>Veranlagungstyp:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_auf_alkohol'">
                <xsl:text>Antrag auf Alkohol</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_rueckerstattungstyp_1_bis_3'">
                <xsl:text>Antrag auf mindestens 2 der R&#252;ckerstattungstyp 1 bis 3</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_biersteuer'">
                <xsl:text>R&#252;ckerstattung Biersteuer</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_andere'">
                <xsl:text>R&#252;ckerstattung andere</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_ausfuhrbeitraeger_lw'">
                <xsl:text>Antrag f&#252;r Ausfuhrbeitr&#228;ge f&#252;r LW</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_voc'">
                <xsl:text>R&#252;ckerstattung der Lenkungsabgabe auf VOC</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sonderfaelle_ex_zil_abfertigungscode_9:'">
                <xsl:text>Sonderf&#228;lle ex ZIL Abfertigungscode 9</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrecht'">
                <xsl:text>Rueckwaren gemaess Zollrecht</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrechtmwstrecht'">
                <xsl:text>Rueckwaren gemaess Zoll-/MWST-Recht</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgabenfreie_abfertigung'">
                <xsl:text>Abgabenfreie Abfertigung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollfreie_abfertigung_gem_zg'">
                <xsl:text>Zollfreie Abfertigung gem. ZG</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabak'">
                <xsl:text>Tabak</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollerleichterung'">
                <xsl:text>Zollerleichterung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckwaren'">
                <xsl:text>R&#252;ckwaren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausbesserungsverkehr'">
                <xsl:text>Ausbesserungsverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veredelungsverkehr'">
                <xsl:text>Veredelungsverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'normalabfertigung'">
                <xsl:text>Normalveranlagung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Besondere Vermerke:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorpapiere'">
                <xsl:text>Vorpapiere:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'container'">
                <xsl:text>Container:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transport'">
                <xsl:text>Transport:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_nr'">
                <xsl:text>MWST-Nr:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'positionen'">
                <xsl:text>Positionen:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transporteur'">
                <xsl:text>Transporteur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dekl_nr'">
                <xsl:text>Dekl.Nr.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'spediteur'">
                <xsl:text>Spediteur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'empfaenger'">
                <xsl:text>Empf&#228;nger:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ref_nr'">
                <xsl:text>Ref.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tin_nr'">
                <xsl:text>Sped-Nr./TIN/UID:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'versender'">
                <xsl:text>Versender:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessische_zollverwaltung_ezv'">
                <xsl:text>Eidgen&#246;ssische Zollverwaltung EZV</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessisches_finanzdepartement_efd'">
                <xsl:text>Eidgen&#246;ssisches Finanzdepartement EFD</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rechtsmittelbelehrung'">
                <xsl:text>Rechtsmittelbelehrung:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstellungsdatum'">
                <xsl:text>Ausstellungsdatum:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annahmedatum'">
                <xsl:text>Annahmedatum:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung_ausfuhr'">
                <xsl:text>VERANLAGUNGSVERF&#252;GUNG AUSFUHR</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung'">
                <xsl:text>Veranlagungsverf&#252;gung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ordentliches_verfahren'">
                <xsl:text>Ordentliches Verfahren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vereinfachtes_verfahren'">
                <xsl:text>Vereinfachtes Verfahren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besonderes_verfahren'">
                <xsl:text>Besonderes Verfahren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annullation'">
                <xsl:text>Annullation</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'verfahren'">
                <xsl:text>Verfahren:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'importeur'">
                <xsl:text>Importeur:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bestimmungsland'">
                <xsl:text>Bestimmungsland:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erzeugungsland'">
                <xsl:text>Versendungsland:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'reason'">
                <xsl:text>Grund</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bordereauNumber'">
                <xsl:text>Bordereau Nr:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto_mwst'">
                <xsl:text>Konto MWST:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto_zoll'">
                <xsl:text>Konto Zoll:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst'">
                <xsl:text>MWST [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mehrwertsteuernummer'">
                <xsl:text>Mehrwertsteuernummer bei R&#252;ckerstattungen:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgaben'">
                <xsl:text>Abgaben gem&#228;ss Veranlagungsverf&#252;gung Zoll:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage'">
                <xsl:text>Bemessungsgrundlage MWST:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage_zoll'">
                <xsl:text>Bemessungsgrundlage:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz_prozent'">
                <xsl:text>Ansatz [%]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Betrag [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kommentar_zu_MWST'">
                <xsl:text>Kommentar zu MWST:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_wert'">
                <xsl:text>MWST Wert:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_rueck'">
                <xsl:text>MWST:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmenart'">
                <xsl:text>Einnahmenart:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage'">
                <xsl:text>Bemessungsgrundlage MWST:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage_zoll'">
                <xsl:text>Bemessungsgrundlage:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz'">
                <xsl:text>Ansatz [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'volumenprozent'">
                <xsl:text>Vol-%:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Betrag [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag'">
                <xsl:text>Gesamtbetrag [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_ohne'">
                <xsl:text>Gesamtbetrag:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zusatzangabenschluessel'">
                <xsl:text>Zusatzangaben-schl&#252;ssel:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettogewicht'">
                <xsl:text>Nettogewicht:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sachbearbeiter'">
                <xsl:text>SachbearbeiterIn:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'referenz_zoll'">
                <xsl:text>Referenz Zoll:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'berichtigungsgrund'">
                <xsl:text>Berichtigungsgrund:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dossiernummer_spediteur'">
                <xsl:text>Referenz Kunde</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kunde'">
                <xsl:text>Kunde:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto'">
                <xsl:text>Konto:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmeart'">
                <xsl:text>Einnahmeart:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettorueckverguetung'">
                <xsl:text>Nettor&#252;ckverg&#252;tung:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anm_nr'">
                <xsl:text>Anmeld.Nr:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'monopolgebuehren'">
                <xsl:text>Monopolgeb&#252;hren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'grenztieraerztliche_untersuchung'">
                <xsl:text>Grenztier&#228;rztliche Untersuchung / Untersuchungsgeb&#252;hr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fauna-gebuehr'">
                <xsl:text>CITES Fauna-Geb&#252;hr (Artenschutz-Kontrolle)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabaksteuer'">
                <xsl:text>Tabaksteuer</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'banderolensteuer'">
                <xsl:text>Banderolensteuer</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'SOTA-gebuehren'">
                <xsl:text>SOTA-Geb&#252;hren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabakpraeventionsfonds'">
                <xsl:text>Tabakpr&#228;ventionsfonds</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'biersteuer'">
                <xsl:text>Biersteuer</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteue_benzin'">
                <xsl:text>Mineral&#246;lsteuer auf Treibstoff Benzin</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_diesel'">
                <xsl:text>Mineral&#246;lsteuer auf Treibstoff Diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_petrol'">
                <xsl:text>Mineral&#246;lsteuer auf Treibstoff Petrol</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_anderen_treibstoffen'">
                <xsl:text>Mineral&#246;lsteuer auf anderen Treibstoffen</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_benzin'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf Treibstoff Benzin</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_diesel'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf Treibstoff Diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_petrol'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf Treibstoff Petrol</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_anderen_treibstoffen'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf anderen Treibstoffen</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_mineraloelprodukten'">
                <xsl:text>Mineral&#246;lsteuer auf Brennstoffen und anderen Mineral&#246;lprodukten</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'automobilsteuer'">
                <xsl:text>Automobilsteuer</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_VOC'">
                <xsl:text>Lenkungsabgabe auf fl&#252;chtigen organischen Verbindungen (VOC)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgaben_heizoel'">
                <xsl:text>Lenkungsabgaben auf Heiz&#246;l extra-leicht</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_autobenzin'">
                <xsl:text>Lenkungsabgabe auf Autobenzin</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_dieseloel'">
                <xsl:text>Lenkungsabgabe auf Diesel&#246;l</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_heizoel'">
                <xsl:text>CO2-Abgabe auf Heiz&#246;l</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_erdgas'">
                <xsl:text>CO2-Abgabe auf Erdgas</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_mineraloelen'">
                <xsl:text>CO2-Abgabe auf anderen Mineral&#246;len</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_kohle'">
                <xsl:text>CO2-Abgabe auf Kohle/Petrolkoks</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollanteil_pflanzenschutzabgabe'">
                <xsl:text>Zollanteil Pflanzenschutzabgabe</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorgezogene_entsorgungsgebuehr'">
                <xsl:text>Vorgezogene Entsorgungsgeb&#252;hr f&#252;r Getr&#228;nkeverpackungen aus Glas</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollabgaben'">
                <xsl:text>Zollabgaben</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch'">
                <xsl:text>provisorisch</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch_with'">
                <xsl:text>provisorisch</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_mwst'">
                <xsl:text>Gesamtbetrag MWST [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz'">
                <xsl:text>Pr&#228;ferenz: </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pkw'">
                <xsl:text>PKW</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkw'">
                <xsl:text>LKW</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwmn'">
                <xsl:text>LKW mit Normalanh&#228;nger</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwms'">
                <xsl:text>LKW mit Sattelanh&#228;nger</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderetm'">
                <xsl:text>Andere</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bahnverkehr'">
                <xsl:text>Bahnverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'andereGebuehren'">
                <xsl:text>Andere Geb&#252;hren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'edelmetallkontrol'">
                <xsl:text>Edelmetallkontrollgeb&#252;hren</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kanzleigebuehr'">
                <xsl:text>Kanzleigeb&#252;hren Grenztierarzt</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provuebersiedlung'">
                <xsl:text>Prov. &#252;bersiedlungsgut</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugbenzin'">
                <xsl:text>Mineral&#246;lsteuer auf Treibstoff Flugbenzin</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugpetrol'">
                <xsl:text>Mineral&#246;lsteuer auf Treibstoff Flugpetrol</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'strassenverkehr'">
                <xsl:text>Strassenverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'luftverkehr'">
                <xsl:text>Luftverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'postverkehr'">
                <xsl:text>Postverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pipelineusw'">
                <xsl:text>Pipeline usw.</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'schiffverkehr'">
                <xsl:text>Schiffverkehr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenantrieb'">
                <xsl:text>Eigenantrieb</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderef1j'">
                <xsl:text>Andere; Frist 1 Jahr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldeu'">
                <xsl:text>Ursprungsnachweis f&#252;r L&#228;nder der EU</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'azmozb'">
                <xsl:text>Andere Zollerm&#228;ssigung oder Zollbefreiung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'voc'">
                <xsl:text>VOC</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'chemieprodukte'">
                <xsl:text>Chemieprodukte</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarierung'">
                <xsl:text>Tarierung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tzs'">
                <xsl:text>Tiere zum Schlachten</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'maschinenanlagen'">
                <xsl:text>Maschinenanlagen</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'montage'">
                <xsl:text>Montage/Einbaukosten</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ozdkd'">
                <xsl:text>OZD/KD-Bewilligung f&#252;r die vor&#252;bergehende Verwendung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uritv'">
                <xsl:text>Unregelm&#228;ssigkeiten im Transitverfahren / Verfahren der vor&#252;bergehenden Verwendung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'agrarprodukte'">
                <xsl:text>Agrarprodukte</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldefta'">
                <xsl:text>Ursprungsnachweis f&#252;r L&#228;nder der EFTA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lvm'">
                <xsl:text>Lagerung von Masseng&#252;ter</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fnnftaer'">
                <xsl:text>Fehlende Nachweisnummer f&#252;r Treibstoffe aus erneuerbaren Rohstoffen</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldafha'">
                <xsl:text>Ursprungsnachweis f&#252;r L&#228;nder der aFHA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfel'">
                <xsl:text>Ursprungsnachweis f&#252;r Entwicklungsl&#228;nder</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ursprungsp'">
                <xsl:text>Ursprungs&#252;berpr&#252;fung </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarifierung'">
                <xsl:text>Tarifierung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uebersiedlungsgut'">
                <xsl:text>&#252;bersiedlungsgut</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstattungsgut'">
                <xsl:text>Ausstattungsgut</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erbschaftsgut'">
                <xsl:text>Erbschaftsgut</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af30t'">
                <xsl:text>Andere; Frist 30 Tage</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2j'">
                <xsl:text>Andere; Frist 2 Jahre</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2m'">
                <xsl:text>Andere; Frist 2 Monate </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af6m'">
                <xsl:text>Andere; Frist 6 Monate </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af3j'">
                <xsl:text>Andere; Frist 3 Jahre</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zusatzmenge'">
                <xsl:text>Zusatzmenge</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz1'">
                <xsl:text>Ja</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz0'">
                <xsl:text>Nein</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'definitiv'">
                <xsl:text>Definitiv</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'space'">
                <xsl:text> </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'import'">
                <xsl:text>Import</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'export'">
                <xsl:text>Export</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugbenz'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf Treibstoff Flugbenzin</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugpetrol'">
                <xsl:text>Mineral&#246;lsteuerzuschlag auf Treibstoff Flugpetrol</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pflanzschutzgeb'">
                <xsl:text>Pflanzenschutzgeb&#252;hr</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollbeguenstkontgeb'">
                <xsl:text>Zollbeg&#252;nstigungskontrollgeb&#252;hr</xsl:text>
            </xsl:when>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="TranslateIT">
        <xsl:param name="label" select="/.."/>
        <xsl:choose>
            <xsl:when test="$label = 'zusatzmenge'">
                <xsl:text>Quantit&#224; suppl.</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'art_nummer_datum'">
                <xsl:text>(genere, numero, data, indicazioni supplementari)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'unterlagen'">
                <xsl:text>Documenti:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'typ_stelle_etc'">
                <xsl:text>(ufficio, numero)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bewilligungen'">
                <xsl:text>Permessi:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anzahl_art_nummer:'">
                <xsl:text>(genere, quantit&#224;, marche)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'packstuecke'">
                <xsl:text>Quantit&#224; colli</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vocmenge'">
                <xsl:text>Quantit&#224; COV:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Menzioni speciali:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenmasse'">
                <xsl:text>Massa netta:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rohmasse'">
                <xsl:text>Massa lorda:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'statwert'">
                <xsl:text>Valore statistico:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattungstyp'">
                <xsl:text>tipo di restituzione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungstyp'">
                <xsl:text>Tipo d'imposizione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_auf_alkohol'">
                <xsl:text>Con domanda per l'alcool</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_rueckerstattungstyp_1_bis_3'">
                <xsl:text>Con domanda per il rimborso per al meno 2 codici da 1 a 3</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_biersteuer'">
                <xsl:text>Rimborso per l'imposta sulla birra</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_andere'">
                <xsl:text>Rimborso altri</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'antrag_ausfuhrbeitraeger_lw'">
                <xsl:text>it_Antrag fuer Ausfuhrbeitraege fuer LW</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckerstattung_voc'">
                <xsl:text>Restituzione della tassa COV</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sonderfaelle_ex_zil_abfertigungscode_9'">
                <xsl:text>casi particolari ex SDI codice di sdoganamento 9</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrecht'">
                <xsl:text>merci di ritorno secondo diritto doganale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckgemaesszollrechtmwstrecht'">
                <xsl:text>merci di ritorno secondo diritto doganale e LIVA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgabenfreie_abfertigung'">
                <xsl:text>Sdoganamento in franchigia da tributi Abfertigung</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollfreie_abfertigung_gem_zg'">
                <xsl:text>Sdoganamento in franchigia da dazio secondo LD</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabak'">
                <xsl:text>Tabacco</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollerleichterung'">
                <xsl:text>Agevolazione doganale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rueckwaren'">
                <xsl:text>merci di ritorno</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausbesserungsverkehr'">
                <xsl:text>Traffico di riparazione</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veredelungsverkehr'">
                <xsl:text>traffico di perfezionamento</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'normalabfertigung'">
                <xsl:text>Sdoganamento normale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besondere_vermerke'">
                <xsl:text>Menzioni speciali:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorpapiere'">
                <xsl:text>Documenti precedenti:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'container'">
                <xsl:text>Container:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transport'">
                <xsl:text>Trasporto:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_nr'">
                <xsl:text>N. IVA.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'positionen'">
                <xsl:text>Posizioni:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'transporteur'">
                <xsl:text>Trasportatore:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dekl_nr'">
                <xsl:text>Dekl.Nr.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'spediteur'">
                <xsl:text>Spedizioniere:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'empfaenger'">
                <xsl:text>Destinatario:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ref_nr'">
                <xsl:text>Ref.:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tin_nr'">
                <xsl:text>N. dello spedizioniere/TIN/UID:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'versender'">
                <xsl:text>Mittente:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessische_zollverwaltung_ezv'">
                <xsl:text>Amministrazione federale delle dogane AFD</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eidgenoessisches_finanzdepartement_efd'">
                <xsl:text>Dipartimento federale delle finanze DFF</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'rechtsmittelbelehrung'">
                <xsl:text>Rimedi giuridici:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstellungsdatum'">
                <xsl:text>Data allestimento:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annahmedatum'">
                <xsl:text>Data accettazione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung_ausfuhr'">
                <xsl:text>DECISIONE D'IMPOSIZIONE ALL'ESPORTAZIONE</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'veranlagungsverfuegung'">
                <xsl:text>Decisione d'imposizione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ordentliches_verfahren'">
                <xsl:text>Procedura ordinaria</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vereinfachtes_verfahren'">
                <xsl:text>Procedura semplificata</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'besonderes_verfahren'">
                <xsl:text>Procedura speciale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'annullation'">
                <xsl:text>Annullamento</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'verfahren'">
                <xsl:text>Procedura</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'importeur'">
                <xsl:text>Importatore</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'reason'">
                <xsl:text>Motivo</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bestimmungsland'">
                <xsl:text>Paese di destinazione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erzeugungsland'">
                <xsl:text>Paese di produzione:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bordereauNumber'">
                <xsl:text>N. della distinta</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'VATAccount'">
                <xsl:text>Conto IVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto_zoll'">
                <xsl:text>Conto doganale:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst'">
                <xsl:text>IVA [CHF]</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mehrwertsteuernummer'">
                <xsl:text>Numero IVA con rimborso</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'abgaben'">
                <xsl:text>Riscossione secondo decisione d’imposizione dogana:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage'">
                <xsl:text>base di calcolo dell’imposta IVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage_zoll'">
                <xsl:text>Base di calcolo:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz_prozent'">
                <xsl:text>Aliquota [%]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Importo [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kommentar_zu_MWST'">
                <xsl:text>Commendo sul IVA:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_wert'">
                <xsl:text>Valore IVA [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mwst_rueck'">
                <xsl:text>IVA [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmenart'">
                <xsl:text>Genere di introiti:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bemessungsgrundlage'">
                <xsl:text>Base di calcolo:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ansatz'">
                <xsl:text>Aliquota [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'volumenprozent'">
                <xsl:text>Vol-%:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'betrag'">
                <xsl:text>Importo [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag'">
                <xsl:text>importo complessivo [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_ohne'">
                <xsl:text>Importo totale:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zusatzangabenschluessel'">
                <xsl:text>Chiave tributi suppletivi:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettogewicht'">
                <xsl:text>Peso netto:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'sachbearbeiter'">
                <xsl:text>Affare trattato da:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'referenz_zoll'">
                <xsl:text>Referenza dogana:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'berichtigungsgrund'">
                <xsl:text>Motivo della rettifica:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'dossiernummer_spediteur'">
                <xsl:text>Numero dossier_spedizioniere</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kunde'">
                <xsl:text>Cliente:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'konto'">
                <xsl:text>Conto:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'einnahmeart'">
                <xsl:text>Genere di introiti:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'nettorueckverguetung'">
                <xsl:text>Restituzione netta:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anm_nr'">
                <xsl:text>Annuncio_Nr:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'monopolgebuehren'">
                <xsl:text>Tasse di monopolio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'grenztieraerztliche_untersuchung'">
                <xsl:text>Visita veterinaria di confine / tassa di visita</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fauna-gebuehr'">
                <xsl:text>Tassa CITES Fauna (controllo sulla conservazione delle specie)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabaksteuer'">
                <xsl:text>Imposta sul tabacco</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'banderolensteuer'">
                <xsl:text>Imposta sulle fascette</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'SOTA-gebuehren'">
                <xsl:text>Tasse SOTA</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tabakpraeventionsfonds'">
                <xsl:text>Fondo per la prevenzione del tabagismo</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'biersteuer'">
                <xsl:text>Imposta sulla birra</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteue_benzin'">
                <xsl:text>Imposta sugli oli minerali gravante la benzina</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_diesel'">
                <xsl:text>Imposta sugli oli minerali gravante il diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_petrol'">
                <xsl:text>Imposta sugli oli minerali gravante il petrolio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_anderen_treibstoffen'">
                <xsl:text>Imposta sugli oli minerali gravante altri carburanti</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_benzin'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante la benzina</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_diesel'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante il diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_petrol'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante il petrolio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuerzuschlag_anderen_treibstoffen'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante altri carburanti</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'mineraloelsteuer_mineraloelprodukten'">
                <xsl:text>Imposta sugli oli minerali gravante i combustibili e altri prodotti a base di oli minerali</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'automobilsteuer'">
                <xsl:text>Imposta sugli autoveicoli</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_VOC'">
                <xsl:text>Tassa d'incentivazione sui composti organici volatili (COV)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgaben_heizoel'">
                <xsl:text>Tasse d'incentivazione sull'olio da riscaldamento extra leggero</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_autobenzin'">
                <xsl:text>Tassa d'incentivazione per benzina per automobili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lenkungsabgabe_dieseloel'">
                <xsl:text>Tasse d'incentivazione sull'olio diesel</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_heizoel'">
                <xsl:text>Tassa sul CO2 sull'olio da riscaldamento</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_erdgas'">
                <xsl:text>Tassa sul CO2 sul gas naturale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_mineraloelen'">
                <xsl:text>Tassa sul CO2 su altri oli minerali</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'CO2-Abgabe_kohle'">
                <xsl:text>Tassa sul CO2 sul carbone/coke di petrolio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollanteil_pflanzenschutzabgabe'">
                <xsl:text>Quota  doganale, tassa fitosanitaria</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'vorgezogene_entsorgungsgebuehr'">
                <xsl:text>Tassa di smaltimento anticipata sugli imballaggi per bevande in vetro</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollabgaben'">
                <xsl:text>tribute doganali</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch'">
                <xsl:text>provvisorio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provisorisch_with'">
                <xsl:text>provvisorio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'gesamtbetrag_mwst'">
                <xsl:text>Importo totale IVA [CHF]:</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz'">
                <xsl:text>Preferenza</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'andereGebuehren'">
                <xsl:text>Altre tasse</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'edelmetallkontrol'">
                <xsl:text>Tasse per il controllo dei metalli preziosi</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'kanzleigebuehr'">
                <xsl:text>Tasse di cancelleria per il veterinario di confine</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'provuebersiedlung'">
                <xsl:text>Prov. Masserizie di immigranti</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugbenzin'">
                <xsl:text>Imposta sugli oli minerali gravante la benzina per aeromobili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestflugpetrol'">
                <xsl:text>Imposta sugli oli minerali gravante il petrolio per aeromobili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugbenz'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante la benzina per aeromobili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pkw'">
                <xsl:text>Autovettura</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkw'">
                <xsl:text>Camion</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwmn'">
                <xsl:text>Camion con rimorchio normale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lkwms'">
                <xsl:text>Trattore con semi-rimorchio (bilico)</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderetm'">
                <xsl:text>Altri</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'bahnverkehr'">
                <xsl:text>Traffico ferroviario</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'strassenverkehr'">
                <xsl:text>Traffico stradale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'luftverkehr'">
                <xsl:text>Traffico aereo</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'postverkehr'">
                <xsl:text>Traffico postale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pipelineusw'">
                <xsl:text>Pipeline, ecc.</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'schiffverkehr'">
                <xsl:text>Traffico navale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'eigenantrieb'">
                <xsl:text>Propulsione propria</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'anderef1j'">
                <xsl:text>Altri; termine 1 anno</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldeu'">
                <xsl:text>Prova d'origine per un paese dell'UE</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'azmozb'">
                <xsl:text>Altre riduzioni dei tributi doganali o franchigia doganale</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'voc'">
                <xsl:text>COV</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'chemieprodukte'">
                <xsl:text>Prodotti chimici</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarierung'">
                <xsl:text>Taratura</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tzs'">
                <xsl:text>Animali da macello</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'maschinenanlagen'">
                <xsl:text>Impianti macchine</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'montage'">
                <xsl:text>Montaggio / costi di montaggio</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ozdkd'">
                <xsl:text> Autorizzazione DGD/DC per l'ammissione temporanea</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uritv'">
                <xsl:text>Irregolarit&#224; nel regime di transito / dell'ammissione temporanea</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'agrarprodukte'">
                <xsl:text>Prodotti agricoli</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldefta'">
                <xsl:text>Prova d'origine per un paese dell'AELS</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'lvm'">
                <xsl:text>Deposito di merci del commercio all'ingrosso</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'fnnftaer'">
                <xsl:text>Manca il numero della prova per carburanti provenienti da materie prime rinnovabili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfldafha'">
                <xsl:text>Prova d'origine per un paese degli aALS</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'usnwfel'">
                <xsl:text>Prova d'origine per paesi in via di sviluppo</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ursprungsp'">
                <xsl:text>Controllo dell'origine </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'tarifierung'">
                <xsl:text>Tarifficazione</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'uebersiedlungsgut'">
                <xsl:text>Masserizie di immigranti</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'ausstattungsgut'">
                <xsl:text>Corredi nuziali</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'erbschaftsgut'">
                <xsl:text>Oggetti ereditati</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af30t'">
                <xsl:text>Altri; termine di 30 giorni</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2j'">
                <xsl:text>Altri; termine di 2 anni</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af2m'">
                <xsl:text>Altri; termine di 2 mesi</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af6m'">
                <xsl:text>Altri; termine di 6 mesi</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'af3j'">
                <xsl:text>Altri; termine di 3 anni</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz1'">
                <xsl:text>Si</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'praeferenz0'">
                <xsl:text>No</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'definitiv'">
                <xsl:text>Definitivo</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'space'">
                <xsl:text> </xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'import'">
                <xsl:text>Importazione</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'export'">
                <xsl:text>Esportazione</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'minoestzusflugpetrol'">
                <xsl:text>Supplemento d'imposta sugli oli minerali gravante il petrolio per aeromobili</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'pflanzschutzgeb'">
                <xsl:text>tassa fitosanitaria</xsl:text>
            </xsl:when>
            <xsl:when test="$label = 'zollbeguenstkontgeb'">
                <xsl:text>Tassa di controllo sulle agevolazioni doganali</xsl:text>
            </xsl:when>
        </xsl:choose>
    </xsl:template>

</xsl:stylesheet><!-- Stylus Studio meta-information - (c)1998-2004. Sonic Software Corporation. All rights reserved.
<metaInformation>
<scenarios ><scenario default="no" name="Scenario1" userelativepaths="yes" externalpreview="yes" url="..\..\..\..\temp\evv_zoll.xml" htmlbaseurl="" outputurl="" processortype="internal" profilemode="0" urlprofilexml="" commandline="" additionalpath="" additionalclasspath="" postprocessortype="none" postprocesscommandline="" postprocessadditionalpath="" postprocessgeneratedext=""/><scenario default="yes" name="Scenario2" userelativepaths="yes" externalpreview="no" url="..\..\..\..\temp\stylesheettests\taxDesVat.xml" htmlbaseurl="" outputurl="" processortype="internal" profilemode="0" urlprofilexml="" commandline="" additionalpath="" additionalclasspath="" postprocessortype="none" postprocesscommandline="" postprocessadditionalpath="" postprocessgeneratedext=""/></scenarios><MapperInfo srcSchemaPath="" srcSchemaRoot="" srcSchemaPathIsRelative="yes" srcSchemaInterpretAsXML="no" destSchemaPath="" destSchemaRoot="" destSchemaPathIsRelative="yes" destSchemaInterpretAsXML="no"/>
</metaInformation>
-->